/*
 +----------------------------------------------------------------------+
 | Swoole                                                               |
 +----------------------------------------------------------------------+
 | Copyright (c) 2012-2015 The Swoole Group                             |
 +----------------------------------------------------------------------+
 | This source file is subject to version 2.0 of the Apache license,    |
 | that is bundled with this package in the file LICENSE, and is        |
 | available through the world-wide-web at the following url:           |
 | http://www.apache.org/licenses/LICENSE-2.0.html                      |
 | If you did not receive a copy of the Apache2.0 license and are unable|
 | to obtain it through the world-wide-web, please send a note to       |
 | license@swoole.com so we can mail you a copy immediately.            |
 +----------------------------------------------------------------------+
 | Author: Tianfeng Han  <rango@swoole.com>                             |
 +----------------------------------------------------------------------+
 */

#pragma once

#include "php_swoole_cxx.h"
#include "swoole_server.h"

#include <unordered_map>
#include <list>
#include <vector>

//--------------------------------------------------------
enum php_swoole_server_callback_type {
    SW_SERVER_CB_onStart,           // master
    SW_SERVER_CB_onBeforeShutdown,  // master
    SW_SERVER_CB_onShutdown,        // master
    SW_SERVER_CB_onWorkerStart,     // worker(event & task)
    SW_SERVER_CB_onWorkerStop,      // worker(event & task)
    SW_SERVER_CB_onBeforeReload,    // manager
    SW_SERVER_CB_onAfterReload,     // manager
    SW_SERVER_CB_onTask,            // worker(task)
    SW_SERVER_CB_onFinish,          // worker(event & task)
    SW_SERVER_CB_onWorkerExit,      // worker(event)
    SW_SERVER_CB_onWorkerError,     // manager
    SW_SERVER_CB_onManagerStart,    // manager
    SW_SERVER_CB_onManagerStop,     // manager
    SW_SERVER_CB_onPipeMessage,     // worker(event & task)
};
//--------------------------------------------------------
enum php_swoole_server_port_callback_type {
    SW_SERVER_CB_onConnect,                  // stream, worker(event)
    SW_SERVER_CB_onReceive,                  // stream, worker(event)
    SW_SERVER_CB_onClose,                    // stream, worker(event)
    SW_SERVER_CB_onPacket,                   // dgram, worker(event)
    SW_SERVER_CB_onRequest,                  // http, worker(event)
    SW_SERVER_CB_onHandshake,                // websocket, worker(event)
    SW_SERVER_CB_onBeforeHandshakeResponse,  // websocket, worker(event)
    SW_SERVER_CB_onOpen,                     // websocket, worker(event)
    SW_SERVER_CB_onMessage,                  // websocket, worker(event)
    SW_SERVER_CB_onDisconnect,               // websocket (non websocket connection), worker(event)
    SW_SERVER_CB_onBufferFull,               // worker(event)
    SW_SERVER_CB_onBufferEmpty,              // worker(event)
};

#define PHP_SWOOLE_SERVER_CALLBACK_NUM (SW_SERVER_CB_onPipeMessage + 1)
#define PHP_SWOOLE_SERVER_PORT_CALLBACK_NUM (SW_SERVER_CB_onBufferEmpty + 1)

namespace swoole {
struct ServerPortProperty;
struct TaskCo;
};  // namespace swoole

zval *php_swoole_server_zval_ptr(swoole::Server *serv);
swoole::ServerPortProperty *php_swoole_server_get_port_property(swoole::ListenPort *port);
void php_swoole_server_set_port_property(swoole::ListenPort *port, swoole::ServerPortProperty *property);

namespace swoole {

struct ServerPortProperty {
    zend::Callable *callbacks[PHP_SWOOLE_SERVER_PORT_CALLBACK_NUM];
    Server *serv;
    ListenPort *port;
    zval *zsetting;
};

struct ServerProperty {
    std::vector<zval *> ports;
    std::vector<zval *> user_processes;
    zend::Callable *callbacks[PHP_SWOOLE_SERVER_CALLBACK_NUM];
    std::unordered_map<TaskId, zend::Callable *> task_callbacks;
    std::unordered_map<TaskId, TaskCo *> task_coroutine_map;
    std::unordered_map<SessionId, std::list<Coroutine *> *> send_coroutine_map;
    std::vector<zend::Callable *> command_callbacks;
};

struct ServerObject {
    Server *serv;
    ServerProperty *property;
    zval init_arguments;
    zend_object std;

    zend_class_entry *get_ce() const {
        return Z_OBJCE_P(php_swoole_server_zval_ptr(serv));
    }

    bool isset_callback(ListenPort *port, int event_type) const {
        return (php_swoole_server_get_port_property(port)->callbacks[event_type] ||
                php_swoole_server_get_port_property(serv->get_primary_port())->callbacks[event_type]);
    }

    bool isset_callback(int event_type) const {
        return property->callbacks[event_type] != nullptr;
    }

    zend::Callable *get_callback(int event_type) const {
        return property->callbacks[event_type];
    }

    zend_bool is_websocket_server() const {
        return instanceof_function(get_ce(), swoole_websocket_server_ce);
    }

    zend_bool is_http_server() const {
        return instanceof_function(get_ce(), swoole_http_server_ce);
    }

    zend_bool is_redis_server() const {
        return instanceof_function(get_ce(), swoole_redis_server_ce);
    }

    void register_callback() const;
    void on_before_start();
    void copy_setting(zval *zsetting) const;
};

struct TaskCo {
    Coroutine *co;
    TaskId *list;
    uint32_t count;
    zval *result;
};
void register_admin_server_commands(Server *serv);
}  // namespace swoole

void php_swoole_server_register_callbacks(swServer *serv);
zend::Callable *php_swoole_server_get_callback(swServer *serv, int server_fd, int event_type);
int php_swoole_create_dir(const char *path, size_t length);
void php_swoole_server_before_start(swServer *serv, zval *zobject);
bool php_swoole_server_isset_callback(swServer *serv, swoole::ListenPort *port, int event_type);
void php_swoole_server_send_yield(swServer *serv, swoole::SessionId sesion_id, zval *zdata, zval *return_value);
void php_swoole_get_recv_data(swServer *serv, zval *zdata, swoole::RecvData *req);
void php_swoole_server_onConnect(swServer *, swoole::DataHead *);
int php_swoole_server_onReceive(swServer *, swoole::RecvData *);
int php_swoole_http_server_onReceive(swServer *, swoole::RecvData *);
void php_swoole_http_server_onClose(swServer *serv, swoole::DataHead *info);
int php_swoole_redis_server_onReceive(swServer *serv, swoole::RecvData *req);
int php_swoole_server_onPacket(swServer *, swoole::RecvData *);
void php_swoole_server_onClose(swServer *, swoole::DataHead *);
void php_swoole_server_onBufferFull(swServer *, swoole::DataHead *);
void php_swoole_server_onBufferEmpty(swServer *, swoole::DataHead *);

swServer *php_swoole_server_get_and_check_server(zval *zobject);
void php_swoole_server_port_deref(zend_object *object);
void php_swoole_server_set_websocket_option(swoole::ListenPort *port, zend_array *vht);
swoole::ServerObject *php_swoole_server_get_zend_object(swoole::Server *serv);
