<?php

namespace DDTrace\Integrations\Kafka;

use DDTrace\HookData;
use DDTrace\Integrations\Integration;
use DDTrace\SpanLink;
use DDTrace\Tag;
use DDTrace\Type;
use DDTrace\Util\ObjectKVStore;

class KafkaIntegration extends Integration
{
    const NAME = 'kafka';

    const METADATA_MAPPING = [
        'metadata.broker.list' => Tag::KAFKA_HOST_LIST,
        'group.id' => Tag::KAFKA_GROUP_ID,
        'client.id' => Tag::KAFKA_CLIENT_ID
    ];

    public static function init(): int
    {
        if (strtok(phpversion('rdkafka'), '.') < 6) {
            return Integration::NOT_LOADED;
        }

        self::installProducerTopicHooks();
        self::installConsumerHooks();
        self::installConfigurationHooks();

        return Integration::LOADED;
    }

    private static function installProducerTopicHooks()
    {
        \DDTrace\install_hook(
            'RdKafka\ProducerTopic::producev',
            static function (HookData $hook) {
                /** @var \RdKafka\ProducerTopic $this */
                KafkaIntegration::setupKafkaProduceSpan($hook, $hook->instance);
            }
        );
    }

    public static function setupKafkaProduceSpan(HookData $hook, \RdKafka\ProducerTopic $producerTopic)
    {
        $span = $hook->span();
        self::setupCommonSpanMetadata($span, Tag::KAFKA_PRODUCE, Tag::SPAN_KIND_VALUE_PRODUCER, Tag::MQ_OPERATION_SEND);

        $span->meta[Tag::MQ_DESTINATION] = $producerTopic->getName();
        $span->meta[Tag::MQ_DESTINATION_KIND] = Type::QUEUE;

        $conf = ObjectKVStore::get($producerTopic, 'conf');
        self::addProducerSpanMetadata($span, $conf, $hook->args);

        if (\ddtrace_config_distributed_tracing_enabled()) {
            $headers = \DDTrace\generate_distributed_tracing_headers();
            $hook->args = self::injectHeadersIntoArgs($hook->args, $headers);
            $hook->overrideArguments($hook->args);
        }
    }

    public static function addProducerSpanMetadata($span, $conf, $args)
    {
        self::addMetadataToSpan($span, $conf);
        $span->metrics[Tag::KAFKA_PARTITION] = $args[0];
        $span->metrics[Tag::MQ_MESSAGE_PAYLOAD_SIZE] = strlen($args[2]);
        if (isset($args[3])) {
            $span->meta[Tag::KAFKA_MESSAGE_KEY] = $args[3];
        }
    }

    private static function injectHeadersIntoArgs(array $args, array $headers): array
    {
        // public RdKafka\ProducerTopic::producev (
        //      integer $partition ,
        //      integer $msgflags ,
        //      string $payload [,
        //      string $key = NULL [,
        //      array $headers = NULL [,
        //      integer $timestamp_ms = NULL [,
        //      string $opaque = NULL
        // ]]]] ) : void
        $argsCount = count($args);
        if ($argsCount >= 5) {
            $args[4] = array_merge($args[4] ?? [], $headers);
        } elseif ($argsCount === 4) {
            $args[] = $headers;
        } elseif ($argsCount === 3) {
            $args[] = null;  // $key
            $args[] = $headers;
        }
        return $args;
    }

    private static function installConsumerHooks()
    {
        $consumerMethods = [
            'RdKafka\KafkaConsumer::consume',
            'RdKafka\Queue::consume'
        ];

        foreach ($consumerMethods as $method) {
            \DDTrace\install_hook(
                $method,
                static function (HookData $hook) {
                    $hook->data['start'] = microtime(true);
                },
                static function (HookData $hook) {
                    /** @var \RdKafka\Message $message */
                    $message = $hook->returned;

                    if ($message) {
                        if ($message->headers && $link = SpanLink::fromHeaders($message->headers)) {
                            if (\dd_trace_env_config('DD_TRACE_KAFKA_DISTRIBUTED_TRACING')) {
                                $span = \DDTrace\start_trace_span($hook->data['start']);
                                \DDTrace\consume_distributed_tracing_headers($message->headers);
                            } else {
                                $span = \DDTrace\start_span($hook->data['start']);
                                $span->links[] = $link;
                            }
                        } else {
                            $span = \DDTrace\start_span($hook->data['start']);
                        }

                        $span->meta[Tag::MQ_DESTINATION] = $message->topic_name;
                        $span->meta[Tag::MQ_DESTINATION_KIND] = Type::QUEUE;
                        $span->metrics[Tag::KAFKA_PARTITION] = $message->partition;
                        $span->metrics[Tag::KAFKA_MESSAGE_OFFSET] = $message->offset;
                        $span->metrics[Tag::MQ_MESSAGE_PAYLOAD_SIZE] = strlen($message->payload ?? '');
                    } else {
                        $span = \DDTrace\start_span($hook->data['start']);
                    }

                    if (!$message || $message->payload === null || $message->err === RD_KAFKA_RESP_ERR__PARTITION_EOF) {
                        $span->meta[Tag::KAFKA_TOMBSTONE] = true;
                    }

                    $hook->data['span'] = $span;
                    KafkaIntegration::setupKafkaConsumeSpan($hook, $hook->instance);
                    \DDTrace\collect_code_origins(1);
                    \DDTrace\close_span();
                }
            );
        }
    }

    public static function setupKafkaConsumeSpan(HookData $hook, $consumer)
    {
        $span = $hook->data['span'];
        self::setupCommonSpanMetadata($span, Tag::KAFKA_CONSUME, Tag::SPAN_KIND_VALUE_CONSUMER, Tag::MQ_OPERATION_RECEIVE);

        $conf = ObjectKVStore::get($consumer, 'conf');
        self::addMetadataToSpan($span, $conf);
    }

    private static function addMetadataToSpan($span, $conf)
    {
        foreach (self::METADATA_MAPPING as $configKey => $tagKey) {
            if (isset($conf[$configKey])) {
                $span->meta[$tagKey] = $conf[$configKey];
            }
        }
    }

    public static function setupCommonSpanMetadata($span, string $name, string $spanKind, string $operation)
    {
        $span->name = $name;
        $span->type = Type::QUEUE;
        $span->meta[Tag::SPAN_KIND] = $spanKind;
        $span->meta[Tag::COMPONENT] = self::NAME;
        $span->meta[Tag::MQ_SYSTEM] = self::NAME;
        $span->meta[Tag::MQ_OPERATION] = $operation;
    }

    private static function installConfigurationHooks()
    {
        $configurationHooks = [
            'RdKafka\KafkaConsumer' => ['__construct'],
            'RdKafka\Producer' => ['__construct', 'newTopic'],
            'RdKafka\Consumer' => ['__construct', 'newQueue']
        ];

        foreach ($configurationHooks as $class => $methods) {
            foreach ($methods as $method) {
                self::installConfigurationHook($class, $method);
            }
        }
    }

    private static function installConfigurationHook(string $class, string $method)
    {
        \DDTrace\hook_method(
            $class,
            $method,
            static function ($This, $scope, $args) use ($method) {
                if ($method === '__construct') {
                    $conf = $args[0];
                    ObjectKVStore::put($This, 'conf', $conf->dump());
                }
            },
            static function ($This, $scope, $args, $returnValue) use ($method) {
                if (in_array($method, ['newTopic', 'newQueue'])) {
                    $conf = ObjectKVStore::get($This, 'conf');
                    ObjectKVStore::put($returnValue, 'conf', $conf);
                }
            }
        );
    }
}
